/**
  ******************************************************************************
  * @file    usart_common_interface.c
  * @author  MCD Application Team
  * @brief   Contains USART HW configuration
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

#include "usart_common_interface.h"
#include "iwdg_interface.h"
#include "wwdg_interface.h"
#include "openbl_usart_cmd.h"

#define DEFAULT_BAUDRATE                  115200            /* Default Baudrate */

void USART_Init(USART_TypeDef * usart, SCI_HandleTypeDef* husart)
{
  husart->Instance                        = usart;
  husart->Init.BaudRate                   = DEFAULT_BAUDRATE;
  husart->Init.WordLength                 = SCI_WORDLENGTH_9B;
  husart->Init.StopBits                   = SCI_STOPBITS_1;
  husart->Init.Parity                     = SCI_PARITY_EVEN;
  husart->Init.Mode                       = SCI_MODE_TX_RX;
  husart->Init.HwFlowCtl                  = SCI_HWCONTROL_NONE;
  husart->Init.OverSampling               = SCI_OVERSAMPLING_16;
  husart->AdvancedInit.AdvFeatureInit     = SCI_ADVFEATURE_AUTOBAUDRATE_INIT;
  husart->AdvancedInit.AutoBaudRateEnable = SCI_ADVFEATURE_AUTOBAUDRATE_ENABLE;     /* Enable Auto Baud Rate detection */
  husart->AdvancedInit.AutoBaudRateMode   = SCI_ADVFEATURE_AUTOBAUDRATE_ONSTARTBIT; /* Auto Baud Rate detection starts from the start bit, send 0x7f from the host side */

  HAL_SCI_Init(husart);

  HAL_SCIEx_SetTxFifoThreshold(husart, SCI_TXFIFO_THRESHOLD_1_8);

  HAL_SCIEx_SetRxFifoThreshold(husart, SCI_RXFIFO_THRESHOLD_1_8);

  HAL_SCIEx_DisableFifoMode(husart);
}

void USART_DeInit(SCI_HandleTypeDef* husart)
{
  HAL_SCI_DeInit(husart);
}

void USART_Read(SCI_HandleTypeDef* husart, uint8_t *buf, uint16_t size)
{
  while (size--)
  {
    while (USART_SR_RXNE != READ_BIT(husart->Instance->SR, USART_SR_RXNE))
    {
      OPENBL_WWDG_Refresh();
      OPENBL_IWDG_Refresh();
    }

    *buf++ = (uint8_t)(READ_BIT(husart->Instance->DR, USART_DR_DR) & 0xFFU);
  }
}

void USART_Write(SCI_HandleTypeDef* husart, uint8_t *buf, uint16_t size)
{
  while (size--)
  {
    while (USART_SR_TXE != READ_BIT(husart->Instance->SR, USART_SR_TXE))
    {
      OPENBL_WWDG_Refresh();
      OPENBL_IWDG_Refresh();
    }

    husart->Instance->DR = ((*buf++) & 0x0FF);
  }

  while (USART_SR_TC != READ_BIT(husart->Instance->SR, USART_SR_TC))
  {
    OPENBL_WWDG_Refresh();
    OPENBL_IWDG_Refresh();
  }

  CLEAR_BIT(husart->Instance->SR, USART_SR_TC);
}

uint8_t USART_ProtocolDetection(SCI_HandleTypeDef* husart)
{
  uint8_t tmp = 0U;

  OPENBL_WWDG_Refresh();
  OPENBL_IWDG_Refresh();

  tmp = husart->Instance->DR & 0xFFU;

  /* Check if the USART is addressed */
  if (SYNC_BYTE == tmp)
  {
    CLEAR_BIT(husart->Instance->CR3, USART_CR3_ABREN);

    /* Aknowledge the host */
    tmp = ACK_BYTE;
    USART_Write(husart, &tmp, 1);

    return 1;
  }

  __HAL_SCI_SEND_AUTOBAUD_REQ(husart);

  return 0;
}

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
